"use client"

import React, { useState, useCallback, useRef } from 'react'
import ReactFlow, { 
  addEdge, 
  Background, 
  Controls, 
  MiniMap,
  useNodesState,
  useEdgesState,
  Panel,
  ReactFlowProvider,
  Node,
  Edge,
  Connection,
  XYPosition
} from 'reactflow'
import 'reactflow/dist/style.css'
import { FlowSidebar } from './FlowSidebar'
import { Button } from '@/components/ui/button'
import { Card } from '@/components/ui/card'
import { Save, Play, X } from 'lucide-react'
import { toast } from 'sonner'

const initialNodes: Node[] = [
  {
    id: '1',
    type: 'input',
    data: { label: 'Start: WhatsApp Message' },
    position: { x: 250, y: 5 },
  },
]

let id = 0
const getId = () => `node_${id++}`

export function FlowBuilder() {
  const reactFlowWrapper = useRef<HTMLDivElement>(null)
  const [nodes, setNodes, onNodesChange] = useNodesState(initialNodes)
  const [edges, setEdges, onEdgesChange] = useEdgesState([])
  const [reactFlowInstance, setReactFlowInstance] = useState<any>(null)

  const onConnect = useCallback((params: Connection) => setEdges((eds) => addEdge(params, eds)), [])

  const onDragOver = useCallback((event: React.DragEvent) => {
    event.preventDefault()
    event.dataTransfer.dropEffect = 'move'
  }, [])

  const onDrop = useCallback(
    (event: React.DragEvent) => {
      event.preventDefault()

      const reactFlowBounds = reactFlowWrapper.current?.getBoundingClientRect()
      const type = event.dataTransfer.getData('application/reactflow')

      if (typeof type === 'undefined' || !type || !reactFlowBounds) {
        return
      }

      const position = reactFlowInstance.project({
        x: event.clientX - reactFlowBounds.left,
        y: event.clientY - reactFlowBounds.top,
      })
      const newNode: Node = {
        id: getId(),
        type,
        position,
        data: { label: `${type} node` },
      }

      setNodes((nds) => nds.concat(newNode))
    },
    [reactFlowInstance, setNodes]
  )

  const onSave = () => {
    if (reactFlowInstance) {
      const flow = reactFlowInstance.toObject()
      console.log('Saving flow:', flow)
      toast.success('Workflow saved successfully!')
    }
  }

  return (
    <div className="flex h-[calc(100vh-12rem)] w-full border rounded-xl overflow-hidden bg-background">
      <ReactFlowProvider>
        <div className="flex-1 relative" ref={reactFlowWrapper}>
          <ReactFlow
            nodes={nodes}
            edges={edges}
            onNodesChange={onNodesChange}
            onEdgesChange={onEdgesChange}
            onConnect={onConnect}
            onInit={setReactFlowInstance}
            onDrop={onDrop}
            onDragOver={onDragOver}
            fitView
          >
            <Controls />
            <MiniMap />
            <Background gap={12} size={1} />
            
            <Panel position="top-right" className="flex gap-2">
              <Button size="sm" variant="outline" className="gap-2 bg-background">
                <Play className="h-4 w-4" />
                Test Flow
              </Button>
              <Button size="sm" className="gap-2" onClick={onSave}>
                <Save className="h-4 w-4" />
                Save Workflow
              </Button>
            </Panel>
          </ReactFlow>
        </div>
        <FlowSidebar />
      </ReactFlowProvider>
    </div>
  )
}
