import { createClient } from "@/lib/supabase/server"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { 
  MessageSquare, 
  Share2, 
  Zap, 
  TrendingUp,
  Activity
} from "lucide-react"

export default async function DashboardPage() {
  const supabase = await createClient()
  const { data: { user } } = await supabase.auth.getUser()

  // Fetch real stats from DB
  const { count: whatsappCount } = await supabase
    .from("agents")
    .select("*", { count: 'exact', head: true })
    .eq("type", "whatsapp")

  const { count: socialCount } = await supabase
    .from("agents")
    .select("*", { count: 'exact', head: true })
    .eq("type", "social")

  const { count: automationCount } = await supabase
    .from("automations")
    .select("*", { count: 'exact', head: true })

  const { data: recentLogs } = await supabase
    .from("logs")
    .select("*")
    .order("created_at", { ascending: false })
    .limit(5)

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-3xl font-bold">Welcome back!</h1>
        <p className="text-muted-foreground">Here's what's happening with your AI agents today.</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
            <CardTitle className="text-sm font-medium">WhatsApp Agents</CardTitle>
            <MessageSquare className="h-4 w-4 text-primary" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{whatsappCount || 0}</div>
            <p className="text-xs text-muted-foreground">Active numbers</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
            <CardTitle className="text-sm font-medium">Social Agents</CardTitle>
            <Share2 className="h-4 w-4 text-primary" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{socialCount || 0}</div>
            <p className="text-xs text-muted-foreground">Connected platforms</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
            <CardTitle className="text-sm font-medium">Total Automations</CardTitle>
            <Zap className="h-4 w-4 text-primary" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{automationCount || 0}</div>
            <p className="text-xs text-muted-foreground">Running flows</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
            <CardTitle className="text-sm font-medium">Messages Handled</CardTitle>
            <TrendingUp className="h-4 w-4 text-primary" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">1,284</div>
            <p className="text-xs text-green-500">+12% from last week</p>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card className="col-span-1">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="h-5 w-5" />
              Recent Activity
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {recentLogs && recentLogs.length > 0 ? (
                recentLogs.map((log) => (
                  <div key={log.id} className="flex items-start gap-3 border-b pb-3 last:border-0">
                    <div className={cn(
                      "mt-1 w-2 h-2 rounded-full",
                      log.level === 'error' ? "bg-destructive" : "bg-primary"
                    )} />
                    <div>
                      <p className="text-sm font-medium">{log.message}</p>
                      <p className="text-xs text-muted-foreground">
                        {new Date(log.created_at).toLocaleString()}
                      </p>
                    </div>
                  </div>
                ))
              ) : (
                <p className="text-sm text-muted-foreground">No recent activity found.</p>
              )}
            </div>
          </CardContent>
        </Card>

        <Card className="col-span-1">
          <CardHeader>
            <CardTitle>Quick Actions</CardTitle>
          </CardHeader>
          <CardContent className="grid grid-cols-2 gap-4">
            <Button variant="outline" className="h-24 flex flex-col gap-2" asChild>
              <a href="/dashboard/whatsapp">
                <MessageSquare className="h-6 w-6" />
                <span>New WhatsApp</span>
              </a>
            </Button>
            <Button variant="outline" className="h-24 flex flex-col gap-2" asChild>
              <a href="/dashboard/automations">
                <Zap className="h-6 w-6" />
                <span>Create Flow</span>
              </a>
            </Button>
            <Button variant="outline" className="h-24 flex flex-col gap-2" asChild>
              <a href="/dashboard/social">
                <Share2 className="h-6 w-6" />
                <span>Connect Social</span>
              </a>
            </Button>
            <Button variant="outline" className="h-24 flex flex-col gap-2" asChild>
              <a href="/dashboard/settings">
                <Settings className="h-6 w-6" />
                <span>Org Settings</span>
              </a>
            </Button>
          </CardContent>
        </Card>
      </div>
    </div>
  )
}

function cn(...inputs: any[]) {
  return inputs.filter(Boolean).join(" ")
}
