import { createClient } from "@/lib/supabase/server"
import { 
  Table, 
  TableBody, 
  TableCell, 
  TableHead, 
  TableHeader, 
  TableRow 
} from "@/components/ui/table"
import { Badge } from "@/components/ui/badge"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Search, Server, ShieldAlert } from "lucide-react"

export default async function AdminLogsPage() {
  const supabase = await createClient()
  const { data: logs } = await supabase
    .from("logs")
    .select("*, organizations(name), agents(name)")
    .order("created_at", { ascending: false })

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-3xl font-bold text-slate-900">System Logs</h1>
          <p className="text-muted-foreground">Global activity monitor across all SaaS tenants.</p>
        </div>
        <div className="flex gap-2">
          <Badge variant="outline" className="bg-blue-50 text-blue-700 border-blue-200 gap-1">
            <Server className="h-3 w-3" />
            System Healthy
          </Badge>
        </div>
      </div>

      <Card>
        <CardHeader className="flex flex-row items-center justify-between">
          <CardTitle>Global Event History</CardTitle>
          <div className="relative">
            <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input placeholder="Filter by organization or message..." className="pl-9 w-[350px]" />
          </div>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Timestamp</TableHead>
                <TableHead>Organization</TableHead>
                <TableHead>Agent</TableHead>
                <TableHead>Level</TableHead>
                <TableHead>Message</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {logs && logs.length > 0 ? (
                logs.map((log) => (
                  <TableRow key={log.id}>
                    <TableCell className="text-xs font-mono whitespace-nowrap">
                      {new Date(log.created_at).toLocaleString()}
                    </TableCell>
                    <TableCell className="font-medium">
                      {log.organizations?.name || "Global"}
                    </TableCell>
                    <TableCell>
                      {log.agents?.name || "System"}
                    </TableCell>
                    <TableCell>
                      <Badge variant={
                        log.level === 'error' ? 'destructive' : 
                        log.level === 'warn' ? 'outline' : 'secondary'
                      } className="text-[10px] uppercase">
                        {log.level}
                      </Badge>
                    </TableCell>
                    <TableCell className="max-w-[400px] truncate text-sm">
                      {log.message}
                    </TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={5} className="text-center py-20 text-muted-foreground">
                    <ShieldAlert className="h-10 w-10 mx-auto mb-2 opacity-20" />
                    No global logs recorded yet.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>
    </div>
  )
}
