'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { AlertCircle } from "lucide-react";

export default function SettingsPage() {
  const { t } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [settings, setSettings] = useState({
    openai_api_key: '',
    model_name: 'gpt-4o',
    ai_behavior: '',
    ai_enabled: true,
  });

  useEffect(() => {
    fetchSettings();
  }, []);

  async function fetchSettings() {
    try {
      const { data, error } = await supabase.from('settings').select('*').single();
      if (error && error.code !== 'PGRST116') throw error;
      if (data) setSettings(data);
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function handleSave() {
    try {
      const { error } = await supabase.from('settings').upsert({
        id: (await supabase.from('settings').select('id').single()).data?.id || undefined,
        ...settings,
        updated_at: new Date().toISOString(),
      });
      if (error) throw error;
      toast.success(t('save') + " successful");
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div>
          <h1 className="text-3xl font-bold tracking-tight text-zinc-900 dark:text-white">
            {t('aiSettings')}
          </h1>
          <p className="text-zinc-500 dark:text-zinc-400">
            Configure your AI behavior and API connections.
          </p>
        </div>

        {!settings.openai_api_key && (
          <div className="flex items-center gap-3 p-4 rounded-lg bg-amber-50 text-amber-800 dark:bg-amber-900/20 dark:text-amber-400 border border-amber-200 dark:border-amber-800">
            <AlertCircle className="h-5 w-5" />
            <p className="text-sm font-medium">{t('noApiKey')}</p>
          </div>
        )}

        <div className="grid gap-6">
          <Card>
            <CardHeader>
              <CardTitle>{t('apiKey')}</CardTitle>
              <CardDescription>Enter your OpenAI API key to enable AI features.</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="apiKey">{t('apiKey')}</Label>
                <Input
                  id="apiKey"
                  type="password"
                  placeholder="sk-..."
                  value={settings.openai_api_key}
                  onChange={(e) => setSettings({ ...settings, openai_api_key: e.target.value })}
                />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>{t('aiSettings')}</CardTitle>
              <CardDescription>Control how the AI behaves and which model it uses.</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between space-x-2">
                <Label htmlFor="ai-enabled" className="flex flex-col space-y-1">
                  <span>Enable AI</span>
                  <span className="font-normal text-xs text-zinc-500">Enable or disable AI auto-responses globally.</span>
                </Label>
                <Switch
                  id="ai-enabled"
                  checked={settings.ai_enabled}
                  onCheckedChange={(val) => setSettings({ ...settings, ai_enabled: val })}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="model">{t('model')}</Label>
                <Select
                  value={settings.model_name}
                  onValueChange={(val) => setSettings({ ...settings, model_name: val })}
                >
                  <SelectTrigger id="model">
                    <SelectValue placeholder="Select a model" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="gpt-4o">GPT-4o</SelectItem>
                    <SelectItem value="gpt-4o-mini">GPT-4o Mini</SelectItem>
                    <SelectItem value="gpt-3.5-turbo">GPT-3.5 Turbo</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="behavior">{t('behavior')}</Label>
                <Textarea
                  id="behavior"
                  placeholder="e.g. You are a helpful assistant for a real estate agency..."
                  className="min-h-[150px]"
                  value={settings.ai_behavior}
                  onChange={(e) => setSettings({ ...settings, ai_behavior: e.target.value })}
                />
              </div>

              <Button onClick={handleSave} className="w-full md:w-auto">
                {t('save')}
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}
