'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { FileText, Plus, Trash2, Database } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";

export default function KnowledgePage() {
  const { t } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [items, setItems] = useState<any[]>([]);
  const [newItem, setNewItem] = useState({ title: '', content: '', type: 'text' });
  const [isDialogOpen, setIsDialogOpen] = useState(false);

  useEffect(() => {
    fetchKnowledge();
  }, []);

  async function fetchKnowledge() {
    try {
      const { data, error } = await supabase.from('knowledge_base').select('*').order('created_at', { ascending: false });
      if (error) throw error;
      setItems(data || []);
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function handleAdd() {
    if (!newItem.title || !newItem.content) return;
    try {
      const { error } = await supabase.from('knowledge_base').insert([newItem]);
      if (error) throw error;
      toast.success("Added successfully");
      setNewItem({ title: '', content: '', type: 'text' });
      setIsDialogOpen(false);
      fetchKnowledge();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function handleDelete(id: string) {
    try {
      const { error } = await supabase.from('knowledge_base').delete().eq('id', id);
      if (error) throw error;
      toast.success("Deleted successfully");
      fetchKnowledge();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-zinc-900 dark:text-white">
              {t('knowledgeTitle')}
            </h1>
            <p className="text-zinc-500 dark:text-zinc-400">
              Manage the data your AI uses to answer questions.
            </p>
          </div>
          
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="mr-2 h-4 w-4" />
                {t('addText')}
              </Button>
            </DialogTrigger>
            <DialogContent className="sm:max-w-[500px]">
                <DialogHeader>
                  <DialogTitle>{t('addText')}</DialogTitle>
                </DialogHeader>
                <div className="space-y-4 py-4">
                  <div className="space-y-2">
                    <Label htmlFor="title">{t('title')}</Label>
                    <Input
                      id="title"
                      placeholder="e.g. Pricing FAQ"
                      value={newItem.title}
                      onChange={(e) => setNewItem({ ...newItem, title: e.target.value })}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="content">{t('content')}</Label>
                    <Textarea
                      id="content"
                      placeholder="Enter the detailed information here..."
                      className="min-h-[200px]"
                      value={newItem.content}
                      onChange={(e) => setNewItem({ ...newItem, content: e.target.value })}
                    />
                  </div>
                  <Button onClick={handleAdd} className="w-full">{t('save')}</Button>
                </div>
              </DialogContent>
            </Dialog>
          </div>
  
          {items.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-24 text-center border-2 border-dashed rounded-xl border-zinc-200 dark:border-zinc-800">
              <Database className="h-12 w-12 text-zinc-300 mb-4" />
              <h3 className="text-lg font-medium text-zinc-900 dark:text-white">{t('noData')}</h3>
              <p className="text-sm text-zinc-500 max-w-xs mx-auto">
                {t('addDataHint')}
              </p>
            </div>
          ) : (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {items.map((item) => (
                <Card key={item.id} className="relative group">
                  <CardHeader className="pb-3">
                    <div className="flex items-start justify-between">
                      <div className="flex items-center gap-2">
                        <FileText className="h-4 w-4 text-zinc-500" />
                        <CardTitle className="text-base font-semibold truncate max-w-[180px]">
                          {item.title}
                        </CardTitle>
                      </div>
                      <Button 
                        variant="ghost" 
                        size="icon" 
                        className="h-8 w-8 text-zinc-400 hover:text-red-500"
                        onClick={() => handleDelete(item.id)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-zinc-600 dark:text-zinc-400 line-clamp-4">
                      {item.content}
                    </p>
                    <div className="mt-4 text-[10px] text-zinc-400 font-medium uppercase tracking-wider">
                      {t('added')} {new Date(item.created_at).toLocaleDateString()}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}

      </div>
    </DashboardLayout>
  );
}
