'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { Search, Send, MessageSquare, Instagram, Facebook, Bot, User } from "lucide-react";
import { cn } from "@/lib/utils";

export default function InboxPage() {
  const { t, isRtl } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [contacts, setContacts] = useState<any[]>([]);
  const [selectedContact, setSelectedContact] = useState<any>(null);
  const [messages, setMessages] = useState<any[]>([]);
  const [newMessage, setNewMessage] = useState('');

  useEffect(() => {
    fetchContacts();
  }, []);

  useEffect(() => {
    if (selectedContact) {
      fetchMessages(selectedContact.id);
    }
  }, [selectedContact]);

  async function fetchContacts() {
    try {
      const { data, error } = await supabase.from('contacts').select('*').order('updated_at', { ascending: false });
      if (error) throw error;
      setContacts(data || []);
      if (data && data.length > 0 && !selectedContact) {
        setSelectedContact(data[0]);
      }
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function fetchMessages(contactId: string) {
    try {
      const { data, error } = await supabase.from('messages').select('*, channels(name, type)').eq('contact_id', contactId).order('created_at', { ascending: true });
      if (error) throw error;
      setMessages(data || []);
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function handleSendMessage() {
    if (!newMessage.trim() || !selectedContact) return;
    try {
      // In a real app, this would trigger an API call to the actual channel (WhatsApp/FB/IG)
      // Here we just save it to the DB as an outbound message from 'agent'
      const { error } = await supabase.from('messages').insert([{
        contact_id: selectedContact.id,
        content: newMessage,
        direction: 'outbound',
        sender_type: 'agent',
        channel_id: (await supabase.from('channels').select('id').eq('type', selectedContact.source).limit(1).single()).data?.id
      }]);
      
      if (error) throw error;
      setNewMessage('');
      fetchMessages(selectedContact.id);
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  const getSourceIcon = (source: string) => {
    switch (source) {
      case 'whatsapp': return <MessageSquare className="h-3 w-3 text-green-500" />;
      case 'facebook': return <Facebook className="h-3 w-3 text-blue-600" />;
      case 'instagram': return <Instagram className="h-3 w-3 text-pink-500" />;
      default: return null;
    }
  };

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="h-[calc(100vh-140px)] flex gap-4">
        {/* Contact List */}
        <div className="w-80 flex flex-col bg-white dark:bg-zinc-900 border rounded-xl overflow-hidden">
            <div className="p-4 border-b">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-zinc-400" />
                <Input placeholder={t('searchConversations')} className="pl-9" />
              </div>
            </div>
            <ScrollArea className="flex-1">
              {contacts.length === 0 ? (
                <div className="p-8 text-center text-zinc-500 text-sm">{t('noConversations')}</div>
              ) : (
                contacts.map((contact) => (

                <button
                  key={contact.id}
                  onClick={() => setSelectedContact(contact)}
                  className={cn(
                    "w-full flex items-center gap-3 p-4 text-left hover:bg-zinc-50 dark:hover:bg-zinc-800 transition-colors",
                    selectedContact?.id === contact.id && "bg-zinc-50 dark:bg-zinc-800 border-l-4 border-zinc-900 dark:border-white"
                  )}
                >
                  <div className="relative">
                    <Avatar>
                      <AvatarFallback>{contact.name?.[0] || '?'}</AvatarFallback>
                    </Avatar>
                    <div className="absolute -bottom-1 -right-1 p-0.5 rounded-full bg-white dark:bg-zinc-900 border">
                      {getSourceIcon(contact.source)}
                    </div>
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between">
                      <p className="font-semibold text-sm truncate">{contact.name || contact.identifier}</p>
                    </div>
                    <p className="text-xs text-zinc-500 truncate">{contact.identifier}</p>
                  </div>
                </button>
              ))
            )}
          </ScrollArea>
        </div>

        {/* Chat Window */}
        <div className="flex-1 flex flex-col bg-white dark:bg-zinc-900 border rounded-xl overflow-hidden">
          {selectedContact ? (
            <>
              <div className="p-4 border-b flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <Avatar>
                    <AvatarFallback>{selectedContact.name?.[0] || '?'}</AvatarFallback>
                  </Avatar>
                  <div>
                    <h3 className="font-semibold text-sm">{selectedContact.name || selectedContact.identifier}</h3>
                    <p className="text-xs text-zinc-500 flex items-center gap-1 uppercase">
                      {getSourceIcon(selectedContact.source)}
                      {selectedContact.source} • {selectedContact.identifier}
                    </p>
                  </div>
                </div>
              </div>

              <ScrollArea className="flex-1 p-4">
                <div className="space-y-4">
                  {messages.map((msg) => (
                    <div
                      key={msg.id}
                      className={cn(
                        "flex flex-col max-w-[80%]",
                        msg.direction === 'outbound' ? (isRtl ? "mr-auto items-start" : "ml-auto items-end") : (isRtl ? "ml-auto items-end" : "mr-auto items-start")
                      )}
                    >
                      <div className={cn(
                        "rounded-2xl px-4 py-2 text-sm",
                        msg.direction === 'outbound' 
                          ? "bg-zinc-900 text-white dark:bg-white dark:text-zinc-900" 
                          : "bg-zinc-100 text-zinc-900 dark:bg-zinc-800 dark:text-white"
                      )}>
                        {msg.content}
                      </div>
                      <div className="mt-1 flex items-center gap-1 px-1 text-[10px] text-zinc-400 uppercase tracking-wider">
                        {msg.sender_type === 'ai' && <Bot className="h-3 w-3" />}
                        {msg.sender_type === 'agent' && <User className="h-3 w-3" />}
                        {msg.sender_type} • {new Date(msg.created_at).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                      </div>
                    </div>
                  ))}
                </div>
              </ScrollArea>

              <div className="p-4 border-t">
                <form 
                  onSubmit={(e) => { e.preventDefault(); handleSendMessage(); }}
                  className="flex gap-2"
                >
                  <Input 
                    placeholder="Type a message..." 
                    value={newMessage}
                    onChange={(e) => setNewMessage(e.target.value)}
                    className="flex-1"
                  />
                  <Button type="submit" size="icon">
                    <Send className="h-4 w-4" />
                  </Button>
                </form>
              </div>
            </>
          ) : (
            <div className="flex-1 flex flex-col items-center justify-center text-zinc-400">
              <MessageSquare className="h-12 w-12 mb-4 opacity-20" />
              <p>Select a conversation to start chatting</p>
            </div>
          )}
        </div>
      </div>
    </DashboardLayout>
  );
}
