'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { Users, Search, Plus, Tag, MessageSquare, Instagram, Facebook, MoreVertical, Trash2 } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";

export default function CRMPage() {
  const { t } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [contacts, setContacts] = useState<any[]>([]);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [newContact, setNewContact] = useState({ name: '', identifier: '', source: 'whatsapp', notes: '', tags: [] as string[] });
  const [tagInput, setTagInput] = useState('');

  useEffect(() => {
    fetchContacts();
  }, []);

  async function fetchContacts() {
    try {
      const { data, error } = await supabase.from('contacts').select('*').order('created_at', { ascending: false });
      if (error) throw error;
      setContacts(data || []);
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function handleAdd() {
    if (!newContact.identifier) return;
    try {
      const { error } = await supabase.from('contacts').insert([newContact]);
      if (error) throw error;
      toast.success("Contact added");
      setNewContact({ name: '', identifier: '', source: 'whatsapp', notes: '', tags: [] });
      setIsDialogOpen(false);
      fetchContacts();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function handleDelete(id: string) {
    try {
      const { error } = await supabase.from('contacts').delete().eq('id', id);
      if (error) throw error;
      toast.success("Contact removed");
      fetchContacts();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  const getSourceIcon = (source: string) => {
    switch (source) {
      case 'whatsapp': return <MessageSquare className="h-4 w-4 text-green-500" />;
      case 'facebook': return <Facebook className="h-4 w-4 text-blue-600" />;
      case 'instagram': return <Instagram className="h-4 w-4 text-pink-500" />;
      default: return null;
    }
  };

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-zinc-900 dark:text-white">
              {t('crm')}
            </h1>
            <p className="text-zinc-500 dark:text-zinc-400">
              Manage your customer relationships across all channels.
            </p>
          </div>
          
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button>
                  <Plus className="mr-2 h-4 w-4" />
                  {t('contacts')}
                </Button>
              </DialogTrigger>
              <DialogContent className="sm:max-w-[500px]">
                <DialogHeader>
                  <DialogTitle>{t('contacts')}</DialogTitle>
                </DialogHeader>
                <div className="space-y-4 py-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="name">{t('name')}</Label>
                      <Input
                        id="name"
                        placeholder="John Doe"
                        value={newContact.name}
                        onChange={(e) => setNewContact({ ...newContact, name: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="source">{t('source')}</Label>
                      <Input
                        id="source"
                        placeholder="whatsapp"
                        value={newContact.source}
                        onChange={(e) => setNewContact({ ...newContact, source: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="identifier">{t('identifier')}</Label>
                    <Input
                      id="identifier"
                      placeholder="+123456789"
                      value={newContact.identifier}
                      onChange={(e) => setNewContact({ ...newContact, identifier: e.target.value })}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="tags">{t('tags')}</Label>
                    <Input
                      id="tags"
                      placeholder="sales, lead, vip"
                      onKeyDown={(e) => {
                        if (e.key === 'Enter') {
                          e.preventDefault();
                          if (tagInput.trim()) {
                            setNewContact({ ...newContact, tags: [...newContact.tags, tagInput.trim()] });
                            setTagInput('');
                          }
                        }
                      }}
                      value={tagInput}
                      onChange={(e) => setTagInput(e.target.value)}
                    />
                    <div className="flex flex-wrap gap-2 mt-2">
                      {newContact.tags.map((tag) => (
                        <Badge key={tag} variant="secondary">{tag}</Badge>
                      ))}
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="notes">{t('reports')}</Label>
                    <Textarea
                      id="notes"
                      placeholder="Customer since 2024..."
                      value={newContact.notes}
                      onChange={(e) => setNewContact({ ...newContact, notes: e.target.value })}
                    />
                  </div>
                  <Button onClick={handleAdd} className="w-full">{t('save')}</Button>
                </div>
              </DialogContent>
            </Dialog>
          </div>
  
          <div className="flex items-center gap-4 bg-white dark:bg-zinc-900 p-4 border rounded-xl">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-zinc-400" />
              <Input placeholder={t('searchConversations')} className="pl-9" />
            </div>
          </div>
  
          <div className="grid gap-4">
            {contacts.length === 0 ? (
              <div className="flex flex-col items-center justify-center py-24 text-center border-2 border-dashed rounded-xl border-zinc-200 dark:border-zinc-800">
                <Users className="h-12 w-12 text-zinc-300 mb-4" />
                <h3 className="text-lg font-medium text-zinc-900 dark:text-white">{t('noContacts')}</h3>
                <p className="text-sm text-zinc-500 max-w-xs mx-auto">
                  {t('addContactHint')}
                </p>
              </div>
            ) : (
              <div className="bg-white dark:bg-zinc-900 border rounded-xl overflow-hidden">
                <table className="w-full text-left text-sm">
                  <thead>
                    <tr className="border-b bg-zinc-50 dark:bg-zinc-800/50">
                      <th className="px-6 py-4 font-semibold">{t('name')}</th>
                      <th className="px-6 py-4 font-semibold">{t('identifier')}</th>
                      <th className="px-6 py-4 font-semibold">{t('source')}</th>
                      <th className="px-6 py-4 font-semibold">{t('tags')}</th>
                      <th className="px-6 py-4 font-semibold">{t('added')}</th>
                      <th className="px-6 py-4 font-semibold"></th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {contacts.map((contact) => (
                      <tr key={contact.id} className="hover:bg-zinc-50 dark:hover:bg-zinc-800/50 transition-colors">
                        <td className="px-6 py-4">
                          <div className="font-medium text-zinc-900 dark:text-white">
                            {contact.name || "Anonymous"}
                          </div>
                        </td>
                        <td className="px-6 py-4 text-zinc-500">
                          {contact.identifier}
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex items-center gap-2 text-zinc-600 dark:text-zinc-400">
                            {getSourceIcon(contact.source)}
                            <span className="capitalize">{contact.source}</span>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex flex-wrap gap-1">
                            {contact.tags?.map((tag: string) => (
                              <Badge key={tag} variant="secondary" className="text-[10px]">{tag}</Badge>
                            ))}
                          </div>
                        </td>
                        <td className="px-6 py-4 text-zinc-400">
                          {new Date(contact.created_at).toLocaleDateString()}
                        </td>
                        <td className="px-6 py-4 text-right">
                          <Button variant="ghost" size="icon" className="text-zinc-400 hover:text-red-500" onClick={() => handleDelete(contact.id)}>
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>

      </div>
    </DashboardLayout>
  );
}
