'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { MessageSquare, Share2, Plus, Trash2, Power, PowerOff, Instagram, Facebook } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";

export default function ChannelsPage() {
  const { t } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [channels, setChannels] = useState<any[]>([]);
  const [newChannel, setNewChannel] = useState({ name: '', type: 'whatsapp', identifier: '', ai_logic: '' });
  const [isDialogOpen, setIsDialogOpen] = useState(false);

  useEffect(() => {
    fetchChannels();
  }, []);

  async function fetchChannels() {
    try {
      const { data, error } = await supabase.from('channels').select('*').order('created_at', { ascending: false });
      if (error) throw error;
      setChannels(data || []);
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function handleAdd() {
    if (!newChannel.name || !newChannel.identifier) return;
    try {
      const { error } = await supabase.from('channels').insert([newChannel]);
      if (error) throw error;
      toast.success("Channel added successfully");
      setNewChannel({ name: '', type: 'whatsapp', identifier: '', ai_logic: '' });
      setIsDialogOpen(false);
      fetchChannels();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function toggleStatus(id: string, currentStatus: string) {
    const newStatus = currentStatus === 'active' ? 'paused' : 'active';
    try {
      const { error } = await supabase.from('channels').update({ status: newStatus }).eq('id', id);
      if (error) throw error;
      toast.success(`Channel ${newStatus}`);
      fetchChannels();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function handleDelete(id: string) {
    try {
      const { error } = await supabase.from('channels').delete().eq('id', id);
      if (error) throw error;
      toast.success("Channel removed");
      fetchChannels();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  const getIcon = (type: string) => {
    switch (type) {
      case 'whatsapp': return <MessageSquare className="h-5 w-5 text-green-500" />;
      case 'facebook': return <Facebook className="h-5 w-5 text-blue-600" />;
      case 'instagram': return <Instagram className="h-5 w-5 text-pink-500" />;
      default: return <Share2 className="h-5 w-5" />;
    }
  };

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-zinc-900 dark:text-white">
              {t('channels')}
            </h1>
            <p className="text-zinc-500 dark:text-zinc-400">
              Connect and manage your WhatsApp numbers and social media accounts.
            </p>
          </div>
          
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="mr-2 h-4 w-4" />
                {t('addChannel')}
              </Button>
            </DialogTrigger>
            <DialogContent className="sm:max-w-[500px]">
                <DialogHeader>
                  <DialogTitle>{t('addChannel')}</DialogTitle>
                </DialogHeader>
                <div className="space-y-4 py-4">
                  <div className="space-y-2">
                    <Label htmlFor="name">{t('channelName')}</Label>
                    <Input
                      id="name"
                      placeholder="e.g. Sales WhatsApp"
                      value={newChannel.name}
                      onChange={(e) => setNewChannel({ ...newChannel, name: e.target.value })}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="type">{t('channelType')}</Label>
                    <Select
                      value={newChannel.type}
                      onValueChange={(val) => setNewChannel({ ...newChannel, type: val })}
                    >
                      <SelectTrigger id="type">
                        <SelectValue placeholder="Select type" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="whatsapp">{t('whatsapp')}</SelectItem>
                        <SelectItem value="facebook">{t('facebook')}</SelectItem>
                        <SelectItem value="instagram">{t('instagram')}</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="identifier">{t('identifier')}</Label>
                    <Input
                      id="identifier"
                      placeholder="e.g. +123456789 or @username"
                      value={newChannel.identifier}
                      onChange={(e) => setNewChannel({ ...newChannel, identifier: e.target.value })}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="logic">{t('aiLogic')}</Label>
                    <Input
                      id="logic"
                      placeholder="e.g. Focus on sales questions"
                      value={newChannel.ai_logic}
                      onChange={(e) => setNewChannel({ ...newChannel, ai_logic: e.target.value })}
                    />
                  </div>
                  <Button onClick={handleAdd} className="w-full">{t('save')}</Button>
                </div>
              </DialogContent>
            </Dialog>
          </div>
  
          <div className="grid gap-6">
            {channels.length === 0 ? (
              <div className="flex flex-col items-center justify-center py-24 text-center border-2 border-dashed rounded-xl border-zinc-200 dark:border-zinc-800">
                <Share2 className="h-12 w-12 text-zinc-300 mb-4" />
                <h3 className="text-lg font-medium text-zinc-900 dark:text-white">{t('noChannels')}</h3>
                <p className="text-sm text-zinc-500 max-w-xs mx-auto">
                  {t('connectFirstChannel')}
                </p>
              </div>
            ) : (

            <div className="grid gap-4 md:grid-cols-2">
              {channels.map((channel) => (
                <Card key={channel.id}>
                  <CardContent className="p-6">
                    <div className="flex items-start justify-between">
                      <div className="flex items-start gap-4">
                        <div className="mt-1 p-2 rounded-lg bg-zinc-100 dark:bg-zinc-800">
                          {getIcon(channel.type)}
                        </div>
                        <div>
                          <div className="flex items-center gap-2">
                            <h3 className="font-semibold text-zinc-900 dark:text-white">{channel.name}</h3>
                            <Badge variant={channel.status === 'active' ? 'default' : 'secondary'} className="text-[10px] uppercase tracking-wider">
                              {channel.status === 'active' ? t('active') : t('paused')}
                            </Badge>
                          </div>
                          <p className="text-sm text-zinc-500">{channel.identifier}</p>
                          {channel.ai_logic && (
                            <p className="mt-2 text-xs text-zinc-400 italic">
                              Logic: {channel.ai_logic}
                            </p>
                          )}
                        </div>
                      </div>
                      
                      <div className="flex items-center gap-2">
                        <Button 
                          variant="ghost" 
                          size="icon" 
                          className={channel.status === 'active' ? "text-amber-500" : "text-green-500"}
                          onClick={() => toggleStatus(channel.id, channel.status)}
                        >
                          {channel.status === 'active' ? <PowerOff className="h-4 w-4" /> : <Power className="h-4 w-4" />}
                        </Button>
                        <Button 
                          variant="ghost" 
                          size="icon" 
                          className="text-zinc-400 hover:text-red-500"
                          onClick={() => handleDelete(channel.id)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </div>
      </div>
    </DashboardLayout>
  );
}
