'use client';

import React, { useEffect, useState } from 'react';
import { DashboardLayout } from "@/components/DashboardLayout";
import { useLanguage } from "@/components/LanguageProvider";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { supabase } from "@/lib/supabase";
import { toast } from "sonner";
import { Zap, Plus, Trash2, ArrowRight } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";

export default function AutomationPage() {
  const { t } = useLanguage();
  const [loading, setLoading] = useState(true);
  const [automations, setAutomations] = useState<any[]>([]);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [newAutomation, setNewAutomation] = useState({ 
    name: '', 
    trigger_type: 'new_message', 
    trigger_value: '', 
    action_type: 'ai_reply', 
    action_value: '' 
  });

  useEffect(() => {
    fetchAutomations();
  }, []);

  async function fetchAutomations() {
    try {
      const { data, error } = await supabase.from('automations').select('*').order('created_at', { ascending: false });
      if (error) throw error;
      setAutomations(data || []);
    } catch (error: any) {
      toast.error(error.message);
    } finally {
      setLoading(false);
    }
  }

  async function handleAdd() {
    if (!newAutomation.name) return;
    try {
      const { error } = await supabase.from('automations').insert([newAutomation]);
      if (error) throw error;
      toast.success("Automation created");
      setNewAutomation({ name: '', trigger_type: 'new_message', trigger_value: '', action_type: 'ai_reply', action_value: '' });
      setIsDialogOpen(false);
      fetchAutomations();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function toggleAutomation(id: string, enabled: boolean) {
    try {
      const { error } = await supabase.from('automations').update({ enabled: !enabled }).eq('id', id);
      if (error) throw error;
      fetchAutomations();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  async function handleDelete(id: string) {
    try {
      const { error } = await supabase.from('automations').delete().eq('id', id);
      if (error) throw error;
      toast.success("Automation deleted");
      fetchAutomations();
    } catch (error: any) {
      toast.error(error.message);
    }
  }

  if (loading) return <DashboardLayout><div>{t('loading')}</div></DashboardLayout>;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-zinc-900 dark:text-white">
              {t('automation')}
            </h1>
            <p className="text-zinc-500 dark:text-zinc-400">
              Create simple rules to automate your responses and routing.
            </p>
          </div>
          
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button>
                  <Plus className="mr-2 h-4 w-4" />
                  {t('automation')}
                </Button>
              </DialogTrigger>
              <DialogContent className="sm:max-w-[500px]">
                <DialogHeader>
                  <DialogTitle>{t('automation')}</DialogTitle>
                </DialogHeader>
                <div className="space-y-4 py-4">
                  <div className="space-y-2">
                    <Label htmlFor="name">{t('automationName')}</Label>
                    <Input
                      id="name"
                      placeholder="e.g. Welcome Message"
                      value={newAutomation.name}
                      onChange={(e) => setNewAutomation({ ...newAutomation, name: e.target.value })}
                    />
                  </div>
                  
                  <div className="grid grid-cols-2 gap-4 border p-4 rounded-lg bg-zinc-50 dark:bg-zinc-800/50">
                    <div className="space-y-2">
                      <Label>{t('when')}</Label>
                      <Select
                        value={newAutomation.trigger_type}
                        onValueChange={(val) => setNewAutomation({ ...newAutomation, trigger_type: val })}
                      >
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="new_message">New Message</SelectItem>
                          <SelectItem value="keyword">Keyword Detected</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    {newAutomation.trigger_type === 'keyword' && (
                      <div className="space-y-2">
                        <Label>Keyword</Label>
                        <Input
                          placeholder="e.g. price"
                          value={newAutomation.trigger_value}
                          onChange={(e) => setNewAutomation({ ...newAutomation, trigger_value: e.target.value })}
                        />
                      </div>
                    )}
                  </div>
  
                  <div className="grid grid-cols-2 gap-4 border p-4 rounded-lg bg-zinc-50 dark:bg-zinc-800/50">
                    <div className="space-y-2">
                      <Label>{t('then')}</Label>
                      <Select
                        value={newAutomation.action_type}
                        onValueChange={(val) => setNewAutomation({ ...newAutomation, action_type: val })}
                      >
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="ai_reply">AI Reply</SelectItem>
                          <SelectItem value="specific_response">Fixed Response</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    {newAutomation.action_type === 'specific_response' && (
                      <div className="space-y-2">
                        <Label>Response</Label>
                        <Input
                          placeholder="e.g. Hello! How can I help?"
                          value={newAutomation.action_value}
                          onChange={(e) => setNewAutomation({ ...newAutomation, action_value: e.target.value })}
                        />
                      </div>
                    )}
                  </div>
  
                  <Button onClick={handleAdd} className="w-full">{t('save')}</Button>
                </div>
              </DialogContent>
            </Dialog>
          </div>
  
          <div className="grid gap-4">
            {automations.length === 0 ? (
              <div className="flex flex-col items-center justify-center py-24 text-center border-2 border-dashed rounded-xl border-zinc-200 dark:border-zinc-800">
                <Zap className="h-12 w-12 text-zinc-300 mb-4" />
                <h3 className="text-lg font-medium text-zinc-900 dark:text-white">{t('noAutomations')}</h3>
                <p className="text-sm text-zinc-500 max-w-xs mx-auto">
                  {t('addAutomationHint')}
                </p>
              </div>
            ) : (

            automations.map((automation) => (
              <Card key={automation.id}>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-6">
                      <div className="p-3 rounded-full bg-zinc-100 dark:bg-zinc-800 text-zinc-900 dark:text-white">
                        <Zap className="h-5 w-5" />
                      </div>
                      <div className="space-y-1">
                        <div className="flex items-center gap-2">
                          <h3 className="font-semibold text-zinc-900 dark:text-white">{automation.name}</h3>
                          {!automation.enabled && <Badge variant="secondary" className="text-[10px] uppercase">Disabled</Badge>}
                        </div>
                        <div className="flex items-center gap-2 text-sm text-zinc-500">
                          <span className="font-medium text-zinc-700 dark:text-zinc-300">
                            {automation.trigger_type === 'new_message' ? 'New Message' : `Keyword: "${automation.trigger_value}"`}
                          </span>
                          <ArrowRight className="h-3 w-3" />
                          <span className="font-medium text-zinc-700 dark:text-zinc-300">
                            {automation.action_type === 'ai_reply' ? 'AI Reply' : 'Fixed Response'}
                          </span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="flex items-center gap-4">
                      <Switch 
                        checked={automation.enabled} 
                        onCheckedChange={() => toggleAutomation(automation.id, automation.enabled)} 
                      />
                      <Button 
                        variant="ghost" 
                        size="icon" 
                        className="text-zinc-400 hover:text-red-500"
                        onClick={() => handleDelete(automation.id)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))
          )}
        </div>
      </div>
    </DashboardLayout>
  );
}
